/* 
 * rt3.h: rt3 trace log related definitions.
 *
 * Copyright (C) Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#ifndef	_RT3_H_
#define	_RT3_H_

#ifdef __KERNEL__
#include <asm/param.h>
#include <asm/system.h>
#ifndef __ASSEMBLY__
#include <linux/hardirq.h>
#include <linux/sched.h>
#include <linux/io.h>
#endif

#include <mach/hardware.h>

#define RT3_TRACE_BUFF_SIZE_IN_WORD \
	((CONFIG_RT3_TRACE_BUFF_SIZE * 1024) / 4) /* sizeof(u32)==4 */

#define TASK_COMM_LEN 16
/* for rt3-trace */
#define RT3_TRACE_DISABLE	0x00000010
#define RT3_TRACE_ENABLE	0x00000011
#define RT3_TRACE_USERLOG	0x00000012
#define RT3_TRACE_USERLOG_EXT	0x00000022
#ifdef CONFIG_RT3_LOG_ISR
#define RT3_TRACE_ENTISR	0x00000013
#define RT3_TRACE_LEAISR	0x00000014
#endif
#define RT3_TRACE_ENTIRQ	0x00000015
#define RT3_TRACE_LEAIRQ	0x00000016
#define RT3_SYS_TRACE_RT3INFO		0x00000017
#define RT3_SYS_TRACE_GETISR		0x00000019
#define RT3_SYS_TRACE_GETBUF		0x00000018
#define RT3_SYS_TRACE_GETNAME		0x00000021
#define RT3_SYS_TRACE_INIT			0x00000023

 
#define RT3_EVE_DISPATCH	0x87000000
#define RT3_EVE_IRQ_ENTRY	0x01000000
#define RT3_EVE_IRQ_EXIT	0x81000000
#ifdef CONFIG_RT3_LOG_ISR
#define RT3_EVE_ISR_ENTRY	0x02000000
#define RT3_EVE_ISR_EXIT	0x82000000
#endif
#define RT3_EVE_LOG_ENABLE	0x0b000000
#define RT3_EVE_LOG_DISABLE	0x9b000000
#define RT3_EVE_USERLOG		0x09000000
#define RT3_EVE_USERLOG8_24	0x19000000
#define RT3_EVE_CREATE		0x26000000
#define RT3_EVE_DELETE		0xa6000000
#define RT3_TRACE_SOFTIRQ_PID	0x8000	/* pseudo pid for softirq */


#define GLOBAL_TIMER_START		(UNIPHIER_PRIVATE_IO_START + 0x200)
#define GLOBAL_TIMER_LOWER_COUNTER	0x0
#define RT3_TIMER ~inl(GLOBAL_TIMER_START)
#define RT3_TRACE_OTHER_CPU_IM	(0x80|raw_smp_processor_id())
#define RT3_TRACE_TIMESTAMP RT3_TIMER
#define RT3_TRACE_TIMESTAMP_SHIFT 4


#ifndef /*************************/ __ASSEMBLY__ /*************************/
typedef struct obs_fname_info {
	ulong	pid;
	char	name[32];
} OBS_FNAME_INFO;

typedef struct rim_info_t {
	ulong	linux_smp;
	ulong	softirq_pid;
	ulong	logid_bitwidth;
	ulong	logid_bitmask;
	ulong	ts_bitwidth;
	ulong	ts_bitmask;
	ulong	val_bitwidth;
	ulong	val_bitmask;
} rim_info;

typedef struct rt3_info_t {
	u32		linux_smp;
	u32		softirq_pid;
	u32		buff;
	u32		buffsize;
	u32		ptr;
	u32		ovrflw;
	u32		enabled;
	u32		timer_cycle;
} rt3_info;

typedef struct rt3_getlog_t {
	u32		ptr;
	u32		logid;
	u32		val;
	u32		ts;
	u32		val2;
} rt3_getlog;

extern u32 rt3_trace(int log_id, int info, int *exinfo);
#define _TRC_usr_log(info) \
	rt3_trace(RT3_TRACE_USERLOG, info, NULL)

#define _TRC_usr_log_ext(info1, info2)				\
	rt3_trace(RT3_TRACE_USERLOG_EXT,		\
		      ((info1) << 24) | ((info2) & 0xffffff),	\
		      NULL)

#ifdef CONFIG_RT3_TRACE_DO_ENABLE
#define rt3_trace_is_enabled()	1	/* always enabled */
#define rt3_trace_enable()		do { } while (0)
#define rt3_trace_disable()		do { } while (0)
#define _TRC_sta_log()
#define _TRC_stp_log()
#else	/* CONFIG_RT3_TRACE_DO_ENABLE */
extern volatile int rt3_trace_enabled;
#define _TRC_sta_log() \
	rt3_trace(RT3_TRACE_ENABLE, 0, NULL)

#define _TRC_stp_log() \
	rt3_trace(RT3_TRACE_DISABLE, 0, NULL)
#ifdef CONFIG_SMP
extern volatile unsigned int rt3_trace_current_pid[NR_CPUS];
#endif /* CONFIG_SMP */
#define rt3_trace_is_enabled()	rt3_trace_enabled
#define rt3_trace_enable() \
	do { rt3_trace_enabled = 1; } while (0)
#define rt3_trace_disable() \
	do { rt3_trace_enabled = 0; } while (0)
#endif	/* CONFIG_RT3_TRACE_DO_ENABLE */

#ifdef CONFIG_RT3_TRACE_LOG_ON_MEM
extern u32 rt3_trace_buff[];
extern volatile u32 *rt3_trace_buff_ptr;
extern volatile int rt3_trace_buff_ovrflw;
#endif

extern unsigned int mn_intc_get_mask_priority(void);

extern unsigned int mn_intc_get_level(unsigned int );

#ifdef CONFIG_SMP
#define rt3_trace_timestamp_cpu(cpuid) \
	((0x00fffe00 & (RT3_TRACE_TIMESTAMP << (8 - RT3_TRACE_TIMESTAMP_SHIFT))) | ((cpuid) << 8))
#define rt3_trace_timestamp()	rt3_trace_timestamp_cpu(raw_smp_processor_id())
#else /* CONFIG_SMP */
#define rt3_trace_timestamp() \
	((0x00ffff00 & (RT3_TRACE_TIMESTAMP << (8 - RT3_TRACE_TIMESTAMP_SHIFT))))
#endif /* CONFIG_SMP */

#ifdef CONFIG_RT3_TRACE_LOG_ON_MEM
extern void rt3_log_event(u32 data);
#else /* CONFIG_RT3_TRACE_LOG_ON_MEM */
#endif /* CONFIG_RT3_TRACE_LOG_ON_MEM */

extern spinlock_t rt3_trace_lock;

#define rt3_trace_spin_lock(lock) _raw_spin_lock(lock)
#define rt3_trace_spin_unlock(lock) _raw_spin_unlock(lock)

#define rt3_trace_spin_lock_irqsave(flags)	\
do {						\
	raw_local_irq_save(flags);		\
	rt3_trace_spin_lock(&rt3_trace_lock);	\
} while (0)

#define rt3_trace_spin_unlock_irqrestore(flags)	\
do {						\
	rt3_trace_spin_unlock(&rt3_trace_lock);	\
	raw_local_irq_restore(flags);		\
} while (0)
/*
 * rt3_log_xxx_event() functions must be used
 * in interrupt disabled context for the consistency of the time-stamp.
 */


static inline void rt3_log_dispatch_event(pid_t ppid, pid_t npid)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_DISPATCH |
			rt3_trace_timestamp());
		rt3_log_event(0xffff & ppid);
	}
#if !defined(CONFIG_RT3_TRACE_DO_ENABLE) && defined(CONFIG_SMP)
	rt3_trace_current_pid[raw_smp_processor_id()] = npid;
#endif /* !defined(CONFIG_RT3_TRACE_DO_ENABLE) && defined(CONFIG_SMP) */
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_irq_entry_event(int ims)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_IRQ_ENTRY |
			 rt3_trace_timestamp() | (0xff & ims));
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_irq_exit_event(int ims)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_IRQ_EXIT |
			rt3_trace_timestamp() | (0xff & ims));
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

#ifdef CONFIG_RT3_LOG_ISR
static inline void rt3_log_isr_entry_event(int isrid_intno)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_ISR_ENTRY |
			rt3_trace_timestamp() | 0x00);
		rt3_log_event(isrid_intno);
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_isr_exit_event(int isrid_intno)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_ISR_EXIT |
			rt3_trace_timestamp() | 0x00);
		rt3_log_event(isrid_intno);
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}
#endif /* CONFIG_RT3_LOG_ISR */


#ifdef CONFIG_RT3_LOG_EXIT_TRACE_LOG
struct rt3_trace_exit_struct {
	pid_t pid;
	char comm[TASK_COMM_LEN];
	long code;
};

extern struct rt3_trace_exit_struct rt3_exit_log_table[CONFIG_RT3_EXIT_TRACE_LOG_MAX_NUM];
extern int rt3_exit_log_cnt;

static inline void rt3_log_process_exit_event(struct task_struct *tsk, long code)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		int idx = rt3_exit_log_cnt;
		if (idx >= CONFIG_RT3_EXIT_TRACE_LOG_MAX_NUM) {
			idx = 0;
		}

		rt3_exit_log_table[idx].pid = tsk->pid;
		memcpy(rt3_exit_log_table[idx].comm, tsk->comm,
		       TASK_COMM_LEN);
		rt3_exit_log_table[idx].code = code;

		rt3_exit_log_cnt = idx + 1;
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}
#endif /* CONFIG_RT3_EXIT_TRACE_LOG */


static inline void rt3_log_user_event(int info)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_USERLOG |
			rt3_trace_timestamp() | (info&0xff));
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_user8_24_event(int info1, int info2)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_USERLOG8_24 |
			rt3_trace_timestamp() | (0xff & info1));
		rt3_log_event(0xffffff & info2);
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_create_event(pid_t pid)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_CREATE |
			rt3_trace_timestamp());
		rt3_log_event(0xffff & pid);
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_delete_event(pid_t pid, int info)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_DELETE |
			rt3_trace_timestamp());
		rt3_log_event(0xffff & pid);
		rt3_log_event(info);
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

#ifndef CONFIG_RT3_TRACE_DO_ENABLE
static inline void rt3_log_enable_event(int ims)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(!rt3_trace_is_enabled())) {
		rt3_trace_enable();
		rt3_log_event(RT3_EVE_LOG_ENABLE |
			rt3_trace_timestamp() | (0xff & ims));
#ifdef CONFIG_SMP
		rt3_log_event(RT3_EVE_LOG_ENABLE |
			rt3_trace_timestamp_cpu(raw_smp_processor_id()^0x1) |
			 (0xff & RT3_TRACE_OTHER_CPU_IM));
#endif /* CONFIG_SMP */
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}

static inline void rt3_log_disable_event(pid_t pid)
{
	unsigned long flags;
	rt3_trace_spin_lock_irqsave(flags);
	if (likely(rt3_trace_is_enabled())) {
		rt3_log_event(RT3_EVE_LOG_DISABLE |
			rt3_trace_timestamp());
		rt3_log_event(0xffff & pid);
#ifdef CONFIG_SMP
		rt3_log_event(RT3_EVE_LOG_DISABLE |
			rt3_trace_timestamp_cpu(raw_smp_processor_id()^0x1));
		rt3_log_event(0xffff & (rt3_trace_current_pid[raw_smp_processor_id()^0x1]));
#endif /* CONFIG_SMP */
		rt3_trace_disable();
	}
	rt3_trace_spin_unlock_irqrestore(flags);
}
#endif /* !CONFIG_RT3_TRACE_DO_ENABLE */

static inline void rt3_trace_switch_to(struct task_struct *prev, struct task_struct *next)
{
	pid_t ppid, npid;

	if (!rt3_trace_is_enabled()) return;
	ppid = prev->pid;
	npid = next->pid;

	if (softirq_count())
		ppid = RT3_TRACE_SOFTIRQ_PID;

	if (softirq_count())
		npid = RT3_TRACE_SOFTIRQ_PID;

	if (ppid == npid)
		return;

	rt3_log_dispatch_event(ppid, npid);
}

#else  /************************* __ASSEMBLY__ *************************/

#endif /************************* __ASSEMBLY__ *************************/
#else /* __KERNEL__ */

#include <sys/ptrace.h>

#define _TRC_sta_log() \
	ptrace((enum __ptrace_request)0x52545310, 0, 0, 0)

#define _TRC_stp_log() \
	ptrace((enum __ptrace_request)0x52545311, 0, 0, 0)

#define _TRC_usr_log(info) \
	ptrace((enum __ptrace_request)0x52545312, 0, 0, info)

#define _TRC_usr_log_ext(info1, info2) \
	ptrace((enum __ptrace_request)0x52545313, 0, 0, \
		((info1) << 24) | ((info2) & 0xffffff))

#endif /* __KERNEL__ */
#endif /* __RT3_H__ */
